/*
GPS coordinate data logger / upload to file / display on google maps
ver 0.6  Stu Sontier 8-8-2012
*/

/*
PURPOSE and OPERATION
Arduino connects to a GPS module, a 16x2 LCD module and an SD card module. It has a (software) debounced switch to change modes 
and an LED to indicate that it has a GPS fix.
Once it has a fix it can be set to start recording GPS data to an SD card.
It can then be told to upload all the data in the file through the serial port.
On the connected computer, a Processing script can grab this data and save to a file on the computer.
A web page can then be reloaded and this will draw a google map with the route information plotted.
*/

/* This code builds on the TinyGPS example.
   It requires the use of SoftwareSerial, SD Card and LiquidCrystal.
  
 PIN CONNECTIONS
   GPS Module:
   9600-baud serial GPS device hooked up on pins 14(rx) and 15(tx).
   
   LCD:
   uses digital pins 3-8 so that serial pins 0 and 1 are free for serial and 2 for interrupt
  
   SD card module:
    ** MOSI - pin 11
    ** MISO - pin 12
    ** CLK - pin 13
    ** CS - pin 10
    
   Switch:
    On digital pin 2
   
   LED:
    On digital pin 9
    
    Operation is dependent on button presses
    1) WAIT - After reset  - LCD shows gps info (and "Waiting" and "press to start") but nothing is logged
    2) LOGGING - If WAIT and button pressed, enter the logging state and log data to CD card. LCD shows gps info and toggles with "Logging" and "press to stop"
    3) STOPPED - If LOGGING and button pressed, enter the stopped state - logging is stopped. LCD shows gps and "Stopped" and "press to upload"
    4) UPLOAD - If STOPPED and button pressed, enter the upload state. SD card read and sent to gobetwino. At end, flash "Upload finished" for 2 seconds then go to WAIT
     * pushbutton attached to pin 2 (interrupt 0) with pullup resistor to +5V
*/

// include the necessary libraries
#include <SoftwareSerial.h>
#include <TinyGPS.h>
#include <LiquidCrystal.h>
#include <SD.h>
#include<stdlib.h>          // for float-to-str
// 
TinyGPS gps;
SoftwareSerial nss(14,15);		// use analog in 0,1 as digital pins for the gps
LiquidCrystal lcd(3, 4, 5, 6, 7, 8);	// leave pins 0,1 available for serial, and pin 2 available for interrupt

// declare prototypes
static void gpsdump(TinyGPS &gps);    
void flashLED(int number); 
void switchInterruptHandler();
void writeToSD(String writeString);
void uploadToPC();
void lcdMessageDisplay(char firstLine[], char secondLine[]);

// variables
const int chipSelect = 10;		// SD card
float flat, flon, gpsSpeed, gpsCourse;
String gpsCourseCard;
unsigned long fix_age;		// gps variables
int sats;				// gps variable
int itemCount = 0;
int debounceTime = 350;    	// the debounce time; increase if the interrupt appears to trigger multiple times
volatile int operatingState = 1;	 //Four possible states 
boolean fixOn = true;
char gotFix = 'X';        // indicates no fix
boolean loopRun = true;  // used to toggle lcd 0,6 to show loop is running

#define BUFSIZ 1002
#define FILENAME "datalog.txt"
//----------------------------------------------
//----------------------------------------------
void setup()
{
  pinMode(9, OUTPUT);            // use this pin to light an led when FIX
  pinMode(10, OUTPUT);          // for chip select on SD module
  pinMode(2, INPUT);		// for the switch
  
  attachInterrupt(0, switchInterruptHandler, FALLING);  // switch on interrupt 0 (pin 2);
  lcd.begin(16, 2);
  
  // Print a message to the LCD.
  lcd.print("Initialising!");
  delay(400);			// a little delay so we can read the LCD
  
  Serial.begin(57600);
  nss.begin(9600);

  //bool stopToggle = FALSE;	// once switch has been pressed for data dump to PC, stop any more data storage (until a reset)
    if (!SD.begin(10)) {
    Serial.println("SD initialization failed!");
    //delay(500);
    return;
  }
  Serial.println("SD initialization okayed!");
  flashLED(12);    // flash the LED a few times to show we have got to the loop and are ready to look for GPS
}

//----------------------------------------------
//----------------------------------------------
void loop()
{
  // Give an indication that the loop is working (toggle a decimal point):
  loopRun = !loopRun;
  lcd.setCursor(6,0);
  (loopRun) ? lcd.print(".") : lcd.print(" ");
  
  // first time through, we're in the WAIT state
  // State only changes at start of each loop (ie interrupt function is only to flag a required state change, not operate on it - this avoids the interrupt causing unexpected behaviour when return fro minterrupt)
    // get GPS data
    bool newData = false;
    unsigned long start = millis();
  
  // For one second we parse GPS data and report some key values
  for (unsigned long start = millis(); millis() - start < 1000;)
  {
    while (nss.available())
    {
      char c = nss.read();
      //Serial.write(c);  // TEST - write GPS data to serial port
      if (gps.encode(c))
      newData = true;     // set a flag when new data available
    }
  }
  // If new data available, check the age of the fix - if longer than 3 secs it is stale data
  if (newData == true)
    {
      gpsdump(gps);
      if (fix_age>0 && fix_age < 3000) 
      {
        gotFix = '0';
        // fix is valid if newdata is true AND fix age is not 0 and less than 3 secs
        fixOn = !fixOn;
        digitalWrite(9, fixOn);   // set the LED to alternate
      }
      else
      {
       // invalid fix
       gotFix = 'X';
       digitalWrite(9, LOW);   // set the LED off
      }
      
      // make string for storing
      char strBuffer[12];
      char ageBuf[12];

      dtostrf(fix_age, 10, 6, ageBuf);
              // avr-gcc      dtostrf(FLOAT,WIDTH,PRECSISION,BUFFER);
        String gpsStr = String("Fix age: ");
        gpsStr +=  ageBuf;
        gpsStr += String("; No " + String(itemCount) + "; Speed: ");
        dtostrf(gps.f_speed_kmph(), 10, 3, strBuffer);        // use this function to convert float to string
        gpsStr += strBuffer;
        gpsStr += "; Lat: ";
        dtostrf(flat, 10, 6, strBuffer);
        gpsStr += strBuffer;
        gpsStr += "; Lon: ";
        dtostrf(flon, 10, 6, strBuffer);
        gpsStr += strBuffer;	// one line of data
        
        // new string - just lat lng comma separated
        gpsStr = "";
        gpsStr += String(itemCount);
        gpsStr += ", ";
        dtostrf(flat, 10, 6, strBuffer);
        gpsStr += strBuffer;
        gpsStr += ", ";
        dtostrf(flon, 10, 6, strBuffer);
        gpsStr += strBuffer;
        gpsStr += ", ";
        dtostrf(gps.f_speed_kmph(), 7, 3, strBuffer);        // use this function to convert float to string
        gpsStr += strBuffer;	// one line of data
        gpsStr += ", ";
        dtostrf(gpsCourse, 10, 6, strBuffer);
        gpsStr += gpsCourseCard;
        
      if (operatingState==2 && fix_age != TinyGPS::GPS_INVALID_AGE) writeToSD(gpsStr);  // if data available and we are in LOG mode, send data to SD
    } 
  
    itemCount +=1;

    // various operations based on state
    switch (operatingState) 
    {
      case (1) :
      
        if (gotFix=='0') {
          lcdMessageDisplay("", "press to log");
        }
        else {
          lcdMessageDisplay("", "Awaiting fix");
        }
        // in WAIT state - Show gps info and LCD message
      	//lcdMessageDisplay("Waiting ", "press to log");
      break;
      case (2) :
        // in LOGGING state, save to SD and LCD message
        // this state is dealt with when new data is found, above
      break;
      case (3) :
        // in STOPPED state - Show gps info and LCD message
      	lcdMessageDisplay("Stopped ", "press to upload");
      break;
      case (4) :
        // in UPLOAD state 
      	uploadToPC();
      break;
    } 
}
//----------------------------------------------
//----------------------------------------------
void switchInterruptHandler()
{
  static unsigned long last_interrupt_time = 0;
  unsigned long interrupt_time = millis();
  // If interrupts come faster than 200ms, assume it's a bounce and ignore
  if (interrupt_time - last_interrupt_time > debounceTime)
  {
    // various opearations based on state
    switch (operatingState) 
    {
      case (1) :
        // if in WAIT state and switch is pressed, change to logging state
        operatingState = 2;
      break;
      case (2) :
        // if in LOGGING state and switch is pressed, change to stopped state
      	operatingState = 3;
      break;
      case (3) :
        // if in STOPPED state and switch is pressed, change to upload state
      	operatingState = 4;
      break;
      case (4) :
        // if in UPLOAD state and switch is pressed, change to wait state
      	operatingState = 1;
      break;
    }
  last_interrupt_time = interrupt_time;    
  }
  else
  {
    //return FALSE;
  }
}
//----------------------------------------------
//----------------------------------------------
  static void gpsdump(TinyGPS &gps)
{
	// gets GPS data; 
  if (gps.satellites()!=TinyGPS::GPS_INVALID_SATELLITES) 
  {
    sats = gps.satellites();
  }
  else
  {
    sats = 0;
  }
  gpsSpeed = gps.f_speed_kmph();
  gpsCourse = gps.f_course();
  gpsCourseCard = TinyGPS::cardinal(gps.f_course());

  gps.f_get_position(&flat, &flon, &fix_age);
}

//----------------------------------------------
//----------------------------------------------
void writeToSD(String dataString)
{
  lcdMessageDisplay(" ", "logging");
  //stop interrupts being able to interrupt
  noInterrupts();

  File dataFile = SD.open(FILENAME, FILE_WRITE);

  // if the file is available, write to it:
  if (dataFile) {
    dataFile.println(dataString);
    dataFile.close();		// finalise the save to SD
    // print to the serial port too:
    //Serial.println(dataString);
  }  
  // if the file isn't open, pop up an error:
  else 
  {
    lcd.setCursor(0,0);
    lcd.print("error opening datalog.txt");
    delay(2000);
  } 
 interrupts();
}
//----------------------------------------------
//----------------------------------------------
void uploadToPC()
{
  lcd.setCursor(0,0);
  lcd.print("uploading");
  Serial.write(FILENAME);
  // re-open the file for reading:
  File myFile = SD.open(FILENAME);
  if (myFile) {
   // Serial.print("filesize:-- ");
    long fileSize = myFile.size();
    String fSize = String(fileSize);
    
    // read from the file until there's nothing else in it:
    while (myFile.available()) {
    	Serial.write(myFile.read());
    }
    
    // write the filesize so Processing knows how many bytes - it can use this line as EOF too
    Serial.write("filesize:-- ");
    //long fileSize = myFile.size();
    Serial.print(fSize);
    Serial.write("\n");
    // close the file:
    myFile.close();
  } else {
  	// if the file didn't open, print an error:
    Serial.println("error opening datalog.txt");
  }
   operatingState = 1;    // go back to WAIT state once file uploaded
}
//----------------------------------------------
//----------------------------------------------
static bool feedgps()
{
  // While there is (softSerial) data available
  while (nss.available())
  {
    if (gps.encode(nss.read()))
      return true;
  }
  return false;
}
//----------------------------------------------
//----------------------------------------------
void lcdMessageDisplay(char firstPart[], char secondPart[])
{
    char buffer[15];
    lcd.clear();
    lcd.setCursor(0,0);

    lcd.print("Speed: ");
    dtostrf(gpsSpeed, 4, 2, buffer);
    lcd.print(buffer);

    lcd.print(" ");
    dtostrf(gpsCourse, 3, 1, buffer);
    lcd.print(gpsCourseCard);   

 // Show gps info, toggling with the 2 lines of the message

   // lcd.clear();
    lcd.setCursor(0,1);
    lcd.print(firstPart);
    //lcd.print(testCount);
    // mode
    lcd.print("[");
    lcd.print(operatingState);
    lcd.print("] ");
    lcd.print(secondPart);
}
//----------------------------------------------
//----------------------------------------------
void flashLED(int number) 
// just to test the led is working
{
 for(int i=0; i<number; i++)
 {
 digitalWrite(9, fixOn);   // set the LED to alternate
 fixOn = !fixOn;
 delay(100);
 }
}
